<?php
declare(strict_types=1);

final class TransfersController {
  /**
   * Auto-accept expired fast transfers for this org.
   * This is intentionally opportunistic (called from /api/inbox and SSE stream)
   * to avoid needing a cron while keeping the UI "unstuck".
   */
  public static function autoAcceptExpiredFastForOrg(string $orgId): void {
    $pdo = db();
    // Find a small batch of expired fast transfers for this org
    $stmt = $pdo->prepare("SELECT id FROM transfer WHERE to_org_id=:oid AND state='pending_fast' AND expires_at IS NOT NULL AND expires_at <= now() ORDER BY expires_at ASC LIMIT 20");
    $stmt->execute([':oid'=>$orgId]);
    $ids = $stmt->fetchAll(PDO::FETCH_COLUMN, 0);
    if (!$ids) return;

    foreach ($ids as $id) {
      // Use the same code path as a normal accept, but mark as auto
      self::acceptTransferInternal((string)$id, null, true);
    }
  }

  public static function inbox(): void {
    $u = require_login();
    $pdo = db();

    // Optional incident scope: lets the UI show only transfers/loans for the currently selected incident.
    $incidentFilter = null;
    $iid = (string)($_GET['incident_id'] ?? '');
    if ($iid && preg_match('/^[a-f0-9\-]{36}$/', $iid)) {
      $incidentFilter = $iid;
      if ($u['role'] !== 'admin') {
        $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
        $chk->execute([':iid'=>$iid, ':oid'=>$u['org_id']]);
        if (!$chk->fetchColumn()) api_error(403, 'not a participant');
      }
    }

    // Opportunistic auto-accept for expired fast-mode transfers
    self::autoAcceptExpiredFastForOrg((string)$u['org_id']);

    // Inbox (open)
    $sql = 'SELECT t.id, t.incident_id, t.type, t.state, t.from_org_id, t.to_org_id, t.mode,
        t.payload_json, t.created_at, t.expires_at,
        o1.name AS from_org_name, o1.short_name AS from_org_short, o2.name AS to_org_name, o2.short_name AS to_org_short
      FROM transfer t
      JOIN org o1 ON o1.id=t.from_org_id
      JOIN org o2 ON o2.id=t.to_org_id
      WHERE t.to_org_id=:oid AND t.state IN (\'pending\',\'pending_fast\')';
    $params = [':oid'=>$u['org_id']];
    if ($incidentFilter) { $sql .= ' AND t.incident_id=:iid'; $params[':iid']=$incidentFilter; }
    $sql .= ' ORDER BY t.created_at DESC LIMIT 200';
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $inbox = $stmt->fetchAll();
    foreach ($inbox as &$r) {
      $r['payload'] = json_decode($r['payload_json'] ?? '{}', true);
      unset($r['payload_json']);
    }

    // Outbox (open)
    $sql = 'SELECT t.id, t.incident_id, t.type, t.state, t.from_org_id, t.to_org_id, t.mode,
        t.payload_json, t.created_at, t.expires_at,
        o1.name AS from_org_name, o1.short_name AS from_org_short, o2.name AS to_org_name, o2.short_name AS to_org_short
      FROM transfer t
      JOIN org o1 ON o1.id=t.from_org_id
      JOIN org o2 ON o2.id=t.to_org_id
      WHERE t.from_org_id=:oid AND t.state IN (\'pending\',\'pending_fast\')';
    $params = [':oid'=>$u['org_id']];
    if ($incidentFilter) { $sql .= ' AND t.incident_id=:iid'; $params[':iid']=$incidentFilter; }
    $sql .= ' ORDER BY t.created_at DESC LIMIT 200';
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $outbox = $stmt->fetchAll();
    foreach ($outbox as &$r) {
      $r['payload'] = json_decode($r['payload_json'] ?? '{}', true);
      unset($r['payload_json']);
    }

    // History (decided/canceled)
    $sql = 'SELECT t.id, t.incident_id, t.type, t.state, t.from_org_id, t.to_org_id, t.mode,
        t.payload_json, t.created_at, t.expires_at, t.decided_at,
        o1.name AS from_org_name, o1.short_name AS from_org_short, o2.name AS to_org_name, o2.short_name AS to_org_short
      FROM transfer t
      JOIN org o1 ON o1.id=t.from_org_id
      JOIN org o2 ON o2.id=t.to_org_id
      WHERE t.from_org_id=:oid AND t.state NOT IN (\'pending\',\'pending_fast\')';
    $params = [':oid'=>$u['org_id']];
    if ($incidentFilter) { $sql .= ' AND t.incident_id=:iid'; $params[':iid']=$incidentFilter; }
    $sql .= ' ORDER BY COALESCE(t.decided_at, t.created_at) DESC LIMIT 200';
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $outbox_history = $stmt->fetchAll();
    foreach ($outbox_history as &$r) {
      $r['payload'] = json_decode($r['payload_json'] ?? '{}', true);
      unset($r['payload_json']);
    }

    $sql = 'SELECT t.id, t.incident_id, t.type, t.state, t.from_org_id, t.to_org_id, t.mode,
        t.payload_json, t.created_at, t.expires_at, t.decided_at,
        o1.name AS from_org_name, o1.short_name AS from_org_short, o2.name AS to_org_name, o2.short_name AS to_org_short
      FROM transfer t
      JOIN org o1 ON o1.id=t.from_org_id
      JOIN org o2 ON o2.id=t.to_org_id
      WHERE t.to_org_id=:oid AND t.state NOT IN (\'pending\',\'pending_fast\')';
    $params = [':oid'=>$u['org_id']];
    if ($incidentFilter) { $sql .= ' AND t.incident_id=:iid'; $params[':iid']=$incidentFilter; }
    $sql .= ' ORDER BY COALESCE(t.decided_at, t.created_at) DESC LIMIT 200';
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $inbox_history = $stmt->fetchAll();
    foreach ($inbox_history as &$r) {
      $r['payload'] = json_decode($r['payload_json'] ?? '{}', true);
      unset($r['payload_json']);
    }

    // Active loans involving my org
    $sql = "SELECT ml.id, ml.incident_id, ml.member_id, ml.from_org_id, ml.to_org_id, ml.status, ml.started_at, ml.ended_at,
        m.display_name AS member_name, m.short_code AS member_code,
        o1.name AS from_org_name, o1.short_name AS from_org_short,
        o2.name AS to_org_name, o2.short_name AS to_org_short
      FROM member_loan ml
      JOIN member m ON m.id=ml.member_id
      JOIN org o1 ON o1.id=ml.from_org_id
      JOIN org o2 ON o2.id=ml.to_org_id
      WHERE ml.status='active' AND (ml.from_org_id=:oid OR ml.to_org_id=:oid)";
    $params = [':oid'=>$u['org_id']];
    if ($incidentFilter) { $sql .= ' AND ml.incident_id=:iid'; $params[':iid']=$incidentFilter; }
    $sql .= ' ORDER BY ml.started_at DESC LIMIT 200';
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $active_loans = $stmt->fetchAll();

    api_json([
      'ok'=>true,
      'incident_id'=>$incidentFilter,
      'inbox'=>$inbox,
      'outbox'=>$outbox,
      'inbox_history'=>$inbox_history,
      'outbox_history'=>$outbox_history,
      'active_loans'=>$active_loans,
    ]);
  }

  public static function endLoan(string $loanId): void {
    $u = require_login();
    $pdo = db();

    db_tx(function(PDO $pdo) use ($loanId, $u) {
      $stmt = $pdo->prepare('SELECT * FROM member_loan WHERE id=:id FOR UPDATE');
      $stmt->execute([':id'=>$loanId]);
      $ml = $stmt->fetch();
      if (!$ml) api_error(404, 'loan not found');
      if ($ml['status'] !== 'active') api_error(400, 'loan already ended');

      $isAdmin = ($u['role'] === 'admin');
      $isParty = ($ml['to_org_id'] === $u['org_id'] || $ml['from_org_id'] === $u['org_id']);
      if (!$isAdmin && !$isParty) api_error(403, 'not allowed');

      // Must be participant of this incident unless admin
      if ($u['role'] !== 'admin') {
        $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
        $chk->execute([':iid'=>$ml['incident_id'], ':oid'=>$u['org_id']]);
        if (!$chk->fetchColumn()) api_error(403, 'not a participant');
      }

      $pdo->prepare("UPDATE member_loan SET status='ended', ended_at=now() WHERE id=:id")
        ->execute([':id'=>$loanId]);

      // If the borrowed member was assigned to a team of the borrowing org, remove the assignment.
      $pdo->prepare('DELETE FROM team_member tm
        USING team t
        WHERE tm.team_id=t.id
          AND t.incident_id=:iid
          AND t.org_id=:to
          AND tm.member_id=:mid')
        ->execute([':iid'=>$ml['incident_id'], ':to'=>$ml['to_org_id'], ':mid'=>$ml['member_id']]);

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([
          ':iid'=>$ml['incident_id'],
          ':uid'=>$u['id'],
          ':act'=>'loan.end',
          ':et'=>'member_loan',
          ':eid'=>$loanId,
          ':m'=>json_encode(['member_id'=>$ml['member_id'],'from_org_id'=>$ml['from_org_id'],'to_org_id'=>$ml['to_org_id']])
        ]);

      EventsController::emitIncident((string)$ml['incident_id'], null, 'all', 'participants.updated', ['incident_id'=>$ml['incident_id']]);
      EventsController::emitIncident((string)$ml['incident_id'], (string)$ml['to_org_id'], 'org', 'inbox.updated', ['incident_id'=>$ml['incident_id']]);
      EventsController::emitIncident((string)$ml['incident_id'], (string)$ml['from_org_id'], 'org', 'inbox.updated', ['incident_id'=>$ml['incident_id']]);
    });

    api_json(['ok'=>true]);
  }

  public static function offerOrder(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $orderId = (string)($in['order_id'] ?? '');
    $toOrgId = (string)($in['to_org_id'] ?? '');
    if (!preg_match('/^[a-f0-9\-]{36}$/', $orderId) || !preg_match('/^[a-f0-9\-]{36}$/', $toOrgId)) {
      api_error(400, 'order_id and to_org_id required');
    }

    $mode = (string)($in['mode'] ?? (public_settings()['ops']['transferModeDefault'] ?? 'offer'));
    $fastGrace = (int)(public_settings()['ops']['fastModeGraceSeconds'] ?? 120);

    $pdo = db();
    $stmt = $pdo->prepare('SELECT id, assigned_org_id, incident_id, title FROM "order" WHERE id=:id AND incident_id=:iid');
    $stmt->execute([':id'=>$orderId, ':iid'=>$incidentId]);
    $o = $stmt->fetch();
    if (!$o) api_error(404, 'order not found');

    if ($toOrgId === $u['org_id']) api_error(400, 'cannot transfer to own org');
    // Target org must be part of this incident
    $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
    $chk->execute([':iid'=>$incidentId, ':oid'=>$toOrgId]);
    if (!$chk->fetchColumn()) api_error(400, 'target org not part of incident');

    // GF can only transfer from own org
    if (!in_array($u['role'], ['admin','el'], true) && $o['assigned_org_id'] !== $u['org_id']) {
      api_error(403, 'cannot transfer order from other org');
    }

    $state = ($mode === 'fast') ? 'pending_fast' : 'pending';
    $expiresAt = null;
    if ($mode === 'fast') {
      $expiresAt = gmdate('c', time() + $fastGrace);
    }

    $tid = null;
    db_tx(function(PDO $pdo) use ($incidentId, $u, $toOrgId, $orderId, $o, $mode, $state, $expiresAt, &$tid) {
      $stmt = $pdo->prepare('INSERT INTO transfer(id, incident_id, type, state, mode, from_org_id, to_org_id, payload_json, expires_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :type, :st, :mode, :from, :to, :p::jsonb, :exp, :uid) RETURNING id');
      $stmt->execute([
        ':iid'=>$incidentId,
        ':type'=>'order_transfer',
        ':st'=>$state,
        ':mode'=>$mode,
        ':from'=>$u['org_id'],
        ':to'=>$toOrgId,
        ':p'=>json_encode(['order_id'=>$orderId, 'order_title'=>$o['title']], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ':exp'=>$expiresAt,
        ':uid'=>$u['id'],
      ]);
      $tid = (string)$stmt->fetchColumn();

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$incidentId, ':uid'=>$u['id'], ':act'=>'transfer.offer', ':et'=>'transfer', ':eid'=>$tid, ':m'=>json_encode(['type'=>'order_transfer','order_id'=>$orderId,'to_org_id'=>$toOrgId,'mode'=>$mode])]);
    });

    // Notify target org
    EventsController::emitIncident($incidentId, $toOrgId, 'org', 'inbox.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$tid], 201);
  }

  public static function offerMemberLoan(string $incidentId): void {
    $u = require_role(['admin','el','gf']);
    $in = json_input();
    $memberId = (string)($in['member_id'] ?? '');
    $toOrgId = (string)($in['to_org_id'] ?? '');
    if (!preg_match('/^[a-f0-9\-]{36}$/', $memberId) || !preg_match('/^[a-f0-9\-]{36}$/', $toOrgId)) {
      api_error(400, 'member_id and to_org_id required');
    }

    if ($toOrgId === $u['org_id']) api_error(400, 'cannot loan to own org');
    $pdo = db();
    $chk = $pdo->prepare('SELECT 1 FROM incident_org WHERE incident_id=:iid AND org_id=:oid');
    $chk->execute([':iid'=>$incidentId, ':oid'=>$toOrgId]);
    if (!$chk->fetchColumn()) api_error(400, 'target org not part of incident');

    // Only from-own-org unless EL/admin
    if (!in_array($u['role'], ['admin','el'], true)) {
      $chk = db()->prepare('SELECT 1 FROM member WHERE id=:id AND org_id=:oid AND is_active=true');
      $chk->execute([':id'=>$memberId, ':oid'=>$u['org_id']]);
      if (!$chk->fetchColumn()) api_error(403, 'member not in your org');
    }

    $mode = (string)($in['mode'] ?? (public_settings()['ops']['transferModeDefault'] ?? 'offer'));
    $fastGrace = (int)(public_settings()['ops']['fastModeGraceSeconds'] ?? 120);
    $state = ($mode === 'fast') ? 'pending_fast' : 'pending';
    $expiresAt = null;
    if ($mode === 'fast') $expiresAt = gmdate('c', time() + $fastGrace);

    $m = $pdo->prepare('SELECT id, display_name, short_code, org_id FROM member WHERE id=:id');
    $m->execute([':id'=>$memberId]);
    $row = $m->fetch();
    if (!$row) api_error(404, 'member not found');

    $tid = null;
    db_tx(function(PDO $pdo) use ($incidentId, $u, $toOrgId, $memberId, $row, $mode, $state, $expiresAt, &$tid) {
      $stmt = $pdo->prepare('INSERT INTO transfer(id, incident_id, type, state, mode, from_org_id, to_org_id, payload_json, expires_at, created_by_user_id)
        VALUES (gen_random_uuid(), :iid, :type, :st, :mode, :from, :to, :p::jsonb, :exp, :uid) RETURNING id');
      $stmt->execute([
        ':iid'=>$incidentId,
        ':type'=>'member_loan',
        ':st'=>$state,
        ':mode'=>$mode,
        ':from'=>$row['org_id'],
        ':to'=>$toOrgId,
        ':p'=>json_encode(['member_id'=>$memberId, 'member_name'=>$row['display_name'], 'member_code'=>$row['short_code']], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
        ':exp'=>$expiresAt,
        ':uid'=>$u['id'],
      ]);
      $tid = (string)$stmt->fetchColumn();
    });

    EventsController::emitIncident($incidentId, $toOrgId, 'org', 'inbox.updated', ['incident_id'=>$incidentId]);
    api_json(['ok'=>true, 'id'=>$tid], 201);
  }

  public static function decide(string $transferId): void {
    $u = require_login();
    $in = json_input();
    $decision = (string)($in['decision'] ?? ''); // accept|reject
    if (!in_array($decision, ['accept','reject'], true)) api_error(400, 'decision must be accept or reject');

    // Authorization: only the target org (or admin) can decide
    $pdo = db();
    $chk = $pdo->prepare('SELECT to_org_id FROM transfer WHERE id=:id');
    $chk->execute([':id'=>$transferId]);
    $toOrg = $chk->fetchColumn();
    if (!$toOrg) api_error(404, 'transfer not found');
    if ($toOrg !== $u['org_id'] && $u['role'] !== 'admin') api_error(403, 'not your inbox');

    if ($decision === 'accept') {
      self::acceptTransferInternal($transferId, (int)$u['id'], false);
    } else {
      self::rejectTransferInternal($transferId, (int)$u['id']);
    }

    api_json(['ok'=>true]);
  }

  public static function cancel(string $transferId): void {
    $u = require_login();

    db_tx(function(PDO $pdo) use ($transferId, $u) {
      $stmt = $pdo->prepare('SELECT * FROM transfer WHERE id=:id FOR UPDATE');
      $stmt->execute([':id'=>$transferId]);
      $t = $stmt->fetch();
      if (!$t) api_error(404, 'transfer not found');
      if ($t['from_org_id'] !== $u['org_id'] && $u['role'] !== 'admin') api_error(403, 'not your outbox');
      if (!in_array($t['state'], ['pending','pending_fast'], true)) api_error(400, 'cannot cancel');

      $pdo->prepare('UPDATE transfer SET state=\'canceled\', decided_at=now(), decided_by_user_id=:uid WHERE id=:id')
        ->execute([':uid'=>$u['id'], ':id'=>$transferId]);

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$t['incident_id'], ':uid'=>$u['id'], ':act'=>'transfer.cancel', ':et'=>'transfer', ':eid'=>$transferId, ':m'=>json_encode(['type'=>$t['type']])]);

      EventsController::emitIncident((string)$t['incident_id'], (string)$t['to_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
    });

    api_json(['ok'=>true]);
  }

  private static function acceptTransferInternal(string $transferId, ?int $actorUserId, bool $isAuto): void {
    db_tx(function(PDO $pdo) use ($transferId, $actorUserId, $isAuto) {
      $stmt = $pdo->prepare('SELECT * FROM transfer WHERE id=:id FOR UPDATE');
      $stmt->execute([':id'=>$transferId]);
      $t = $stmt->fetch();
      if (!$t) return; // auto accept may race with manual decide
      if (!in_array($t['state'], ['pending','pending_fast'], true)) return;

      $payload = json_decode($t['payload_json'] ?? '{}', true) ?: [];
      if ($t['type'] === 'order_transfer') {
        $orderId = (string)($payload['order_id'] ?? '');
        if ($orderId) {
          $pdo->prepare('UPDATE "order" SET assigned_org_id=:to, assigned_team_id=NULL, updated_at=now() WHERE id=:oid')
            ->execute([':to'=>$t['to_org_id'], ':oid'=>$orderId]);
        }
      }
      if ($t['type'] === 'member_loan') {
        $memberId = (string)($payload['member_id'] ?? '');
        if ($memberId) {
          $pdo->prepare('INSERT INTO member_loan(id, incident_id, member_id, from_org_id, to_org_id, transfer_id, status)
            VALUES (gen_random_uuid(), :iid, :mid, :from, :to, :tid, :st)
            ON CONFLICT (incident_id, member_id, to_org_id) DO NOTHING')
            ->execute([':iid'=>$t['incident_id'], ':mid'=>$memberId, ':from'=>$t['from_org_id'], ':to'=>$t['to_org_id'], ':tid'=>$t['id'], ':st'=>'active']);
        }
      }

      $pdo->prepare('UPDATE transfer SET state=\'accepted\', decided_at=now(), decided_by_user_id=:uid WHERE id=:id')
        ->execute([':uid'=>$actorUserId, ':id'=>$transferId]);

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([
          ':iid'=>$t['incident_id'],
          ':uid'=>$actorUserId,
          ':act'=>'transfer.accept',
          ':et'=>'transfer',
          ':eid'=>$transferId,
          ':m'=>json_encode(['auto'=>$isAuto,'type'=>$t['type']])
        ]);

      EventsController::emitIncident((string)$t['incident_id'], (string)$t['to_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      // Also nudge the sender org so Outbox/History updates immediately.
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'transfer.decided', ['incident_id'=>$t['incident_id'], 'transfer_id'=>$transferId, 'state'=>'accepted']);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'orders.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'teams.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], null, 'all', 'participants.updated', ['incident_id'=>$t['incident_id']]);
    });
  }

  private static function rejectTransferInternal(string $transferId, int $actorUserId): void {
    db_tx(function(PDO $pdo) use ($transferId, $actorUserId) {
      $stmt = $pdo->prepare('SELECT * FROM transfer WHERE id=:id FOR UPDATE');
      $stmt->execute([':id'=>$transferId]);
      $t = $stmt->fetch();
      if (!$t) api_error(404, 'transfer not found');
      if (!in_array($t['state'], ['pending','pending_fast'], true)) api_error(400, 'already decided');

      $pdo->prepare('UPDATE transfer SET state=\'rejected\', decided_at=now(), decided_by_user_id=:uid WHERE id=:id')
        ->execute([':uid'=>$actorUserId, ':id'=>$transferId]);

      $pdo->prepare('INSERT INTO audit_log(id, incident_id, actor_user_id, action, entity_type, entity_id, meta_json)
        VALUES (gen_random_uuid(), :iid, :uid, :act, :et, :eid, :m::jsonb)')
        ->execute([':iid'=>$t['incident_id'], ':uid'=>$actorUserId, ':act'=>'transfer.reject', ':et'=>'transfer', ':eid'=>$transferId, ':m'=>json_encode(['type'=>$t['type']])]);

      EventsController::emitIncident((string)$t['incident_id'], (string)$t['to_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'inbox.updated', ['incident_id'=>$t['incident_id']]);
      EventsController::emitIncident((string)$t['incident_id'], (string)$t['from_org_id'], 'org', 'transfer.decided', ['incident_id'=>$t['incident_id'], 'transfer_id'=>$transferId, 'state'=>'rejected']);
    });
  }
}
